<?php

defined( 'ABSPATH' ) or exit( 'Hey, what are you doing here? You silly human!' );

if ( ! class_exists( 'Dusky_Analytics' ) ) {
    class Dusky_Analytics {
        private static $instance = null;

        public function __construct() {

            $is_pro = dusky_fs()->can_use_premium_code__premium_only();
            $is_analytics = dusky_get_settings( 'enableAnalytics', false );

            if ( ! $is_pro || ( $is_pro && $is_analytics ) ) {
                add_action( 'dusky_add_submenu_page', [$this, 'add_analytics_menu'] );
            }

            if ( $is_pro && $is_analytics ) {
                add_action( 'wp_dashboard_setup', [$this, 'dashboard_widgets'] );
                add_action( 'wp_ajax_dusky_filter_dashboard_usage', [$this, 'dusky_filter_dashboard_usage'] );
                add_action( 'wp_ajax_nopriv_dusky_filter_dashboard_usage', [$this, 'dusky_filter_dashboard_usage'] );
                add_action( 'wp_ajax_frontend_toggle', [$this, 'frontend_toggle'] );
                add_action( 'wp_ajax_nopriv_frontend_toggle', [$this, 'frontend_toggle'] );
                add_action( 'wp_ajax_get_analytics', [$this, 'get_analytics_data'] );
                add_action( 'wp_ajax_nopriv_get_analytics', [$this, 'get_analytics_data'] );
                add_action( 'wp', [$this, 'track_visitors'] );
            }
        }

        public function add_analytics_menu( Dusky_Admin $admin ) {
            $admin->add_dusky_submenu( 'analytics',
                'dusky-dark-mode',
                __( 'Analytics - Dusky Dark Mode', 'dusky-dark-mode' ),
                __( 'Analytics', 'dusky-dark-mode' ),
                'manage_options',
                'dusky-analytics',
                [
                    $this,
                    'render_dusky_analytics',
                ]
            );
        }

        public function render_dusky_analytics() {
            echo '<div id="dusky-analytics-wrap"></div>';
        }

        // Define the AJAX callback function
        public function dusky_filter_dashboard_usage() {
            $nonce = isset( $_POST['data'] ) ? sanitize_key( $_POST['data'] ) : null;

            if ( ! wp_verify_nonce( $nonce, 'dusky_nonce' ) ) {
                wp_send_json_error( ['message' => 'You are not authorize'] );
            }

            $data = isset( $_POST['data'] ) ? $this->calculate_dusky_usage( sanitize_text_field( $_POST['data'] ) ) : null;

            wp_send_json_success( $data );
        }

        private function calculate_dusky_usage( $length = 7 ) {

            $all_visitors = (array) get_option( 'dusky_visitors', [gmdate( 'd-m-Y' ) => []] );
            $usages = (array) get_option( 'dusky_usage', [gmdate( 'd-m-Y' ) => 0] );

            $visits = array_slice( $all_visitors, -$length, $length, true );

            $values = [];
            $labels = [];

            if ( ! empty( $visits ) ) {
                foreach ( $visits as $date => $visit ) {
                    $visit = count( $visit );

                    $usage = ! empty( $usages[$date] ) ? $usages[$date] : 0;

                    if ( $visit < 0 ) {
                        $visit = 0;
                    }

                    if ( $usage < 0 ) {
                        $usage = 0;
                    }

                    $labels[] = $date;

                    if ( ! is_numeric( $visit ) || ! is_numeric( $usage ) || 0 === $visit || 0 === $usage ) {
                        $values[] = 0;
                    } else {
                        $values[] = ceil( ( intval( $usage ) / intval( $visit ) ) * 100 );
                    }
                }
            }

            if ( count( $labels ) < $length ) {
                $dates = $labels;
                $zeros = [];
                $dateArray = [];

                $date = DateTime::createFromFormat( 'd-m-Y', $dates[0] );

                for ( $i = 0; $i < ( $length - count( $dates ) ); $i++ ) {
                    $date->modify( '-1 day' );

                    $dateArray[] = $date->format( 'd-m-Y' );
                    $zeros[] = 0;
                }
                $dateArray = array_reverse( $dateArray );
                $labels = array_merge( $dateArray, $dates );
                $values = array_merge( $zeros, $values );
            }

            if ( ! dusky_fs()->can_use_premium_code__premium_only() ) {

                $today = new DateTime();

                $dateArray = [];

                for ( $i = 0; $i < $length; $i++ ) {
                    $dateArray[] = $today->format( 'd-m-Y' );

                    $today->modify( '-1 day' );
                }

                $dateArray = array_reverse( $dateArray );

                $labels = $dateArray;

                $values = ['90', '56', '35', '59', '57', '30', '86'];
            }

            if ( $labels[0] === 0 ) {
                $labels[0] = gmdate( 'd-m-Y' );
            }

            $style = dusky_get_settings( 'chartStyle', 'bar' );

            return [$labels, $values, $style];
        }

        public function dashboard_widgets() {
            $analytics = dusky_get_settings( 'enableAnalytics', false );
            $dashboard_widget = dusky_get_settings( 'dashboardWidget', false );

            if ( ! $analytics || ! $dashboard_widget ) {
                return;
            }

            wp_add_dashboard_widget(
                'dusky_dashboard_widget',
                esc_html__( 'Dusky Dark Mode Usage', 'dusky-dark-mode' ),
                [$this, 'dashboard_widget_cb']
            );

            global $wp_meta_boxes;

            $default_dashboard = $wp_meta_boxes['dashboard']['normal']['core'];

            $dusky_dashboard_widget_backup = ['dusky_dashboard_widget' => $default_dashboard['dusky_dashboard_widget']];
            unset( $default_dashboard['dusky_dashboard_widget'] );

            $sorted_dashboard = array_merge( $dusky_dashboard_widget_backup, $default_dashboard );

            $wp_meta_boxes['dashboard']['normal']['core'] = $sorted_dashboard;
        }

        public function dashboard_widget_cb() {
            wp_enqueue_script( 'dusky-chart' );
            ?>
            <div id="dusky-dark-mode-chart" class="dusky-dark-mode-chart">
                <div class="chart-header">
                    <span><?php esc_html_e( 'How much percentage of users use dark mode each day.', 'dusky-dark-mode' );?></span>

                    <select name="dusky_chart_period" id="dusky_chart_period">
                        <option value="7"><?php esc_html_e( 'Last 7 Days', 'dusky-dark-mode' );?></option>
                        <option value="30"><?php esc_html_e( 'Last 30 Days', 'dusky-dark-mode' );?></option>
                    </select>
                </div>

                <div class="chart-container">
                    <canvas id="dusky_dark_mode_chart" height="300"></canvas>
                </div>

                <?php if ( ! dusky_fs()->can_use_premium_code__premium_only() ) {?>
                    <div class="dusky-dark-mode-chart-modal-wrapper">
                        <div class="dusky-dark-mode-chart-modal">
                            <h2><?php esc_html_e( 'View Dark Mode usages inside WordPress Dashboard', 'dusky-dark-mode' );?></h2>
                            <p><?php esc_html_e( 'Upgrade to Pro and get access to the reports.', 'dusky-dark-mode' );?></p>
                            <p><a href="#" class="button-primary button-hero" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Upgrade to Pro', 'dusky-dark-mode' );?></a>
                            </p>
                        </div>
                    </div>
                <?php }?>

            </div>
        <?php
}

        private function get_visitor_ip() {
            if ( isset( $_SERVER['HTTP_CLIENT_IP'] ) ) {
                return $_SERVER['HTTP_CLIENT_IP'];
            } elseif ( isset( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
                return $_SERVER['HTTP_X_FORWARDED_FOR'];
            } elseif ( isset( $_SERVER['HTTP_X_FORWARDED'] ) ) {
                return $_SERVER['HTTP_X_FORWARDED'];
            } elseif ( isset( $_SERVER['HTTP_FORWARDED_FOR'] ) ) {
                return $_SERVER['HTTP_FORWARDED_FOR'];
            } elseif ( isset( $_SERVER['HTTP_FORWARDED'] ) ) {
                return $_SERVER['HTTP_FORWARDED'];
            } else {
                return $_SERVER['REMOTE_ADDR'];
            }
        }

        private function _track_visitors() {
            $analytics = dusky_get_settings( 'enableAnalytics', false );

            if ( ! $analytics ) {
                return;
            }

            $today = gmdate( 'd-m-Y' );
            $endOfDayTimestamp = strtotime( 'tomorrow midnight' ) - 1;
            $usage = get_option( 'dusky_usage', [] );
            $visitor_ip = $this->get_visitor_ip();
            $visitor_key = md5( $visitor_ip );

            $visitors_key = get_option( 'dusky_visitors', [$today => []] );

            if ( isset( $visitors_key[$today] ) && ! in_array( $visitor_key, $visitors_key[$today] ) ) {
                $visitors_key[$today][] = $visitor_key;
                update_option( 'dusky_visitors', $visitors_key );

            } else if ( ! isset( $visitors_key[$today] ) ) {
                $visitors_key[$today][] = $visitor_key;
                update_option( 'dusky_visitors', $visitors_key );
            }

            if ( isset( $_COOKIE['dusky-front-mode'] ) && sanitize_key( $_COOKIE['dusky-front-mode'] ) === 'dark' && ! isset( $_COOKIE['dusky_used'] ) && ! is_admin() ) {
                if ( ! isset( $usage[$today] ) ) {
                    $usage[$today] = 1;
                } else {
                    $usage[$today]++;
                }
                update_option( 'dusky_usage', $usage );

                setcookie( 'dusky_used', 'used', $endOfDayTimestamp, '/' );
            } elseif ( ! isset( $usage[$today] ) ) {
                $usage[$today] = 0;
                update_option( 'dusky_usage', $usage );
            }
        }

        private function get_unique_id() {
            $visitor_ip = $this->get_visitor_ip();
            return md5( $visitor_ip );
        }

        public function track_visitors() {

            $analytics = dusky_get_settings( 'enableAnalytics', false );
            $isFrontDark = dusky_get_settings( 'frontDark', false );

            if ( ! $analytics || is_admin() || ! $isFrontDark ) {
                return;
            }

            $unique_id = $this->get_unique_id();
            $user = $this->get_user_by_unique_id( $unique_id, true );

            if ( empty( $user ) ) {
                return $this->set_user_analytics( $unique_id );
            } else {
                return $this->update_user_analytics( $user->id, [
                    'view'      => $user->view + 1,
                    'dark_view' => dusky_get_mode() === 'dark' ? $user->dark_view + 1 : $user->dark_view,
                ], [
                    '%d',
                    '%d',
                ] );
            }

            return false;
        }

        public function frontend_toggle() {
            $nonce = isset( $_POST['nonce'] ) ? sanitize_key( $_POST['nonce'] ) : null;
            if ( ! wp_verify_nonce( $nonce, 'dusky_nonce' ) ) {
                wp_send_json_error( ['message' => 'You are not in actual way!'] );
            }

            $analytics = dusky_get_settings( 'enableAnalytics', false );
            $isFrontDark = dusky_get_settings( 'frontDark', false );

            if ( ! $analytics || ! $isFrontDark ) {
                wp_send_json_error( ['analytics' => $analytics, 'isFrontDark' => $isFrontDark] );
            }

            $mode = isset( $_POST['mode'] ) ? sanitize_key( $_POST['mode'] ) : null;

            $unique_id = $this->get_unique_id();
            $user = $this->get_user_by_unique_id( $unique_id, true );

            $data = null;

            if ( $mode === 'dark' ) {
                $data = [
                    'activation'  => $user->activation + 1,
                    'active_mode' => 1,
                ];
            } elseif ( $mode === 'light' ) {
                $data = [
                    'deactivation' => $user->deactivation + 1,
                    'active_mode'  => 0,
                ];
            }

            if ( $user && $data ) {
                $res = $this->update_user_analytics( $user->id, $data, [
                    '%d',
                    '%d',
                ] );

                wp_send_json_success( ['data' => $res] );
            }
            wp_send_json_error( ['error' => $data, 'mode' => $mode, 'user' => $user] );

        }

        private function set_user_analytics( $unique_id ) {
            global $wpdb;

            $table_name = $wpdb->prefix . 'dusky_analytics';

            $data_to_insert = [
                'unique_id' => $unique_id,
                'view'      => 1,
                'dark_view' => dusky_get_mode() === 'dark' && dusky_get_settings( 'frontDark', false ) ? 1 : 0,
                'date'      => current_time( 'mysql', 1 ),
            ];

            $data_format = [
                '%s',
                '%d',
                '%d',
                '%s',
            ];

            return $wpdb->insert( $table_name, $data_to_insert, $data_format );

        }

        private function update_user_analytics( $user_id, $data_to_update, $data_format ) {
            global $wpdb;

            $table_name = $wpdb->prefix . 'dusky_analytics';

            $where_condition = [
                'id' => $user_id,
            ];

            $where_format = [
                '%s',
            ];

            return $wpdb->update( $table_name, $data_to_update, $where_condition, $data_format, $where_format );

        }

        private function get_user_by_unique_id( $id, $isToday = false ) {
            global $wpdb;

            $table_name = $wpdb->prefix . 'dusky_analytics';

            if ( $isToday ) {
                $today = gmdate( 'Y-m-d' );
                return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_name WHERE unique_id = %s AND DATE(date) = %s", $id, $today ) );
            } else {
                return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_name WHERE unique_id = %s", $id ) );
            }
        }

        public function get_analytics() {
            global $wpdb;

            return $wpdb->get_results( "SELECT * FROM {$wpdb->prefix}dusky_analytics" );

        }

        public function get_analytics_data() {
            if ( dusky_fs()->can_use_premium_code__premium_only() ) {
                wp_send_json_success( ['data' => $this->get_analytics()] );
            } else {
                wp_send_json_success( ['data' => [
                    [
                        'id'           => 0,
                        'view'         => 0,
                        'dark_view'    => 0,
                        'active_mode'  => 0,
                        'activation'   => 0,
                        'deactivation' => 0,
                        'unique_id'    => "",
                    ],
                ]] );
            }
        }

        public function get_analytics_by_date( $date_to_retrieve = null, $end_date = null ) {
            global $wpdb;

            if ( is_null( $date_to_retrieve ) ) {
                $date_to_retrieve = gmdate( 'Y-m-d' );
            }

            $table_name = $wpdb->prefix . 'dusky_analytics';

            if ( ! $end_date ) {
                return $wpdb->get_results( $wpdb->prepare( "SELECT * FROM $table_name WHERE DATE(date) = %s", $date_to_retrieve ) );
            } else {
                return $wpdb->get_results( $wpdb->prepare( "SELECT * FROM $table_name WHERE DATE(date) BETWEEN %s AND %s", $date_to_retrieve, $end_date ) );
            }
        }

        public function get_count_users() {
            global $wpdb;

            $table_name = $wpdb->prefix . 'dusky_analytics';

            return $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(DISTINCT unique_id) FROM $table_name" ) );
        }

        public static function instance() {
            if ( is_null( self::$instance ) ) {
                self::$instance = new self();
            }

            return self::$instance;
        }
    }

}

Dusky_Analytics::instance();